/*------------------------------------------------------------------------
  **** BEGIN LICENSE BLOCK ****
  Version: MPL 1.1/GPL 2.0/LGPL 2.1
  Copyright(c) 2004-2010 Created by Eric Fredericksen (www.pttpsystems.com)
  All Rights Reserved.

  This program are subject to the Mozilla Public License Version
  1.1 (the "License"); you may not use this file except in compliance with
  the License. You may obtain a copy of the License at
  http://www.mozilla.org/MPL/

  This program is free software. Software distributed under the License
  is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
  either express or implied. See the License for the specific language
  governing rights and limitations under the License.

  Alternatively, the contents of this file may be used under the terms of
  either the GNU General Public License Version 2 or later (the "GPL"), or
  the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
  in which case the provisions of the GPL or the LGPL are applicable instead
  of those above. If you wish to allow use of your version of this file only
  under the terms of either the GPL or the LGPL, and not to allow others to
  use your version of this file under the terms of the MPL, indicate your
  decision by deleting the provisions above and replace them with the notice
  and other provisions required by the GPL or the LGPL. If you do not delete
  the provisions above, a recipient may use your version of this file under
  the terms of any one of the MPL, the GPL or the LGPL.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
  **** END LICENSE BLOCK ****
------------------------------------------------------------------------*/
#include "stdafx.h"

#include "Anagrams.h"
#include "ThreadTools.h"

#include <sys/timeb.h>

//==================================================
//
//==================================================
CAnagramSignature::CAnagramSignature(LPCTSTR _ptcString)
{
	Signify(_ptcString);
}

//==================================================
//
//==================================================
void CAnagramSignature::Signify(LPCTSTR _ptcString)
{
	RtlZeroMemory(&m_oData.m_factSignature, sizeof(m_oData.m_factSignature) );

	if( !_ptcString ) return;

	for(LPCTSTR ptcChar = _ptcString; *ptcChar; ptcChar++)
	{
		TCHAR tChar = *ptcChar;
		if( tChar >= 'a' && tChar <= 'z' )
		{
			DWORD dwIndex = ((DWORD)tChar)-(DWORD)'a';
			m_oData.m_factSignature[dwIndex]++;
		}
		else if( tChar >= 'A' && tChar <= 'Z' )
		{
			DWORD dwIndex = 26 + ((DWORD)tChar)-(DWORD)'Z';
			m_oData.m_factSignature[dwIndex]++;
		}
		else
		{	// stop if not in [a-zA-Z]
			return;
		}
	}
}


//==================================================
// must define this for using CAnagramSignature as a map<> key
//==================================================
bool operator<( const CAnagramSignature & _roLeft, const CAnagramSignature & _roRight)
{
	for( long K=0; K<CAnagramSignature::DWORD_COUNT; K++)
	{
		if( _roLeft.m_oData.m_fadwSignature[K] < _roRight.m_oData.m_fadwSignature[K] ) return(true);
		if( _roLeft.m_oData.m_fadwSignature[K] > _roRight.m_oData.m_fadwSignature[K] ) return(false);
		// else equal, keep looking
	}

	return(false);
}

//==================================================
//
//==================================================
CAnagramDictionary::CAnagramDictionary(LPCTSTR _ptcFileName)
{
	InitializeCriticalSection(&m_oCriticalSection);

	if( _ptcFileName ) LoadWordList(_ptcFileName);
}

//==================================================
//
//==================================================
CAnagramDictionary::~CAnagramDictionary()
{
	DeleteCriticalSection(&m_oCriticalSection);
}

//==================================================
//
//==================================================
string CAnagramDictionary::ThreadSafeLookup(const CAnagramSignature & _roSignature)
{
	// protect this from multiple threads
	CScopeProtector oScopeProtector(m_oCriticalSection);

	// note that there is an implicit conversion of the string
	// to a CAnagramSignature object inside the []
	string & roszList = m_oSignatureMap[_roSignature];
	if( 0 >= roszList.size() )
	{
		// !!string not present
		// Clear the entry that we entered cause it does not belong
		// Note that we *could* learn new words/strings along the way, but that is not yet required
		m_oSignatureMap[_roSignature].erase();
		return("<NO ENTRY>");
	}
	// Copying strings is more costly but inherently safe.
	// It would be more efficient to return a const char pointer if 
	// thread safety is externally implemented. However, the lookup
	// method inserts into the map automagically, so we can't use
	// the "read-only" nature for thread safety.
	return(roszList);
}

//==================================================
//
//==================================================
bool CAnagramDictionary::LoadWordList(LPCTSTR _ptcFileName)
{
	struct _timeb oStart, oEnd;
	_ftime_s( &oStart);

	FILE * poInput = NULL; 
	errno_t tError = fopen_s(&poInput, _ptcFileName, "r");
	if( !poInput ) return(false);

	char fatcInputLine[MAX_LINEREAD_BUFFER];

	while( !feof(poInput) && !ferror(poInput) )
	{
		fgets(fatcInputLine, MAX_STRING_LENGTH, poInput);
		if( ferror(poInput) ) continue;
		// null terminate for safety
		fatcInputLine[MAX_STRING_LENGTH]=0;

		// kill the terminating newline
		char * pcNL = strchr(fatcInputLine, '\n');
		if(pcNL) *pcNL = 0;

		string & roString = m_oSignatureMap[fatcInputLine];
		if( 0 < roString.size() ) 
		{
			//NB: 2010/03/24 I changed the "\r\n" into "; " because the output is easer to read :)
			roString += "; ";
		}
		roString += fatcInputLine;
	}

	_ftime_s( &oEnd);
	m_fLoadTime = oEnd.time*1.0f + oEnd.millitm/1000.0f - oStart.time*1.0f - oStart.millitm/1000.0f;

	fprintf(stderr,_T("CAnagramDictionary::LoadWordList load time %.3f seconds\n"), m_fLoadTime);
	return(true);
}


//==================================================
//
//==================================================
CWordList::CWordList(LPCTSTR _ptcFileName)
: m_bWordListLoaded(false)
{
	InitializeCriticalSection(&m_oCriticalSection);
	if( _ptcFileName )
	{
		m_bWordListLoaded = LoadWordList(_ptcFileName);
		if( !m_bWordListLoaded )
		{
			// avoid an empty list!
			m_oWordList.push_back("AAAARRRRGGGGHHHH");
		}
	}
}

//==================================================
//
//==================================================
CWordList::~CWordList()
{
	DeleteCriticalSection(&m_oCriticalSection);
}

//==================================================
//
//==================================================
bool CWordList::LoadWordList(LPCTSTR _ptcFileName)
{
	fprintf(stderr,_T("Loading words[%s]...\n"), _ptcFileName);
	struct _timeb oStart, oEnd;
	_ftime_s( &oStart);

	FILE * poInput = NULL;
	errno_t tError = fopen_s(&poInput, _ptcFileName, "r");
	if( !poInput )
	{
		fprintf(stderr,_T("Could not load the word list! [%u] -- something bad is about to happen. :(\n"), tError);
		return(false);
	}

	char fatcInputLine[MAX_LINEREAD_BUFFER];
	m_oWordList.clear();

	while( !feof(poInput) && !ferror(poInput) )
	{
		fgets(fatcInputLine, MAX_STRING_LENGTH, poInput);
		if( ferror(poInput) ) continue;
		// null terminate for safety
		fatcInputLine[MAX_STRING_LENGTH]=0;

		// kill the terminating newline
		char * pcNL = strchr(fatcInputLine, '\n');
		if(pcNL) *pcNL = 0;
		// we presume uniqueness in our word list
		m_oWordList.push_back( fatcInputLine );
	}

	_ftime_s( &oEnd);
	// nice - changing to 64-bit affects the code -- the small differences get rounded by the large *.time number
	// solution: rearrange like a good numerical analyst to subtract numbers of like magnitude :)
	//m_fLoadTime = oEnd.time*1.0f + oEnd.millitm/1000.0f - oStart.time*1.0f - oStart.millitm/1000.0f;
	m_fLoadTime = (oEnd.time-oStart.time)*1.0f + (oEnd.millitm-oStart.millitm)/1000.0f;

	fprintf(stderr, _T("CWordList::LoadWordList load time %.3f seconds\n"), m_fLoadTime);

	return(true);
}

//==================================================
//
//==================================================
string CWordList::ThreadSafeLookup(DWORD _dwIndex)
{
	// protect this from multiple threads
	CScopeProtector oScopeProtector(m_oCriticalSection);

	// make the index modulo the length
	_dwIndex = _dwIndex % m_oWordList.size();

	string & roszEntry = m_oWordList[ _dwIndex ];
	// Copying strings is more costly but inherently safe.
	// It would be more efficient to return a const char pointer if 
	// thread safety is externally implemented. Also, we may be able to use
	// the "read-only" nature for thread safety. as long as the structure does not change.
	return(roszEntry);
}
